package com.example.jsr268client;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.List;

import android.app.Activity;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.net.Uri;
import android.os.AsyncTask;
import android.os.Bundle;
import android.smartcardio.ATR;
import android.smartcardio.Card;
import android.smartcardio.CardException;
import android.smartcardio.CardNotPresentException;
import android.smartcardio.CardTerminal;
import android.smartcardio.TerminalFactory;
import android.smartcardio.ipc.CardService;
import android.smartcardio.ipc.ICardService;
import android.util.Log;
import android.view.Gravity;
import android.view.Menu;
import android.view.View;
import android.widget.Button;
import android.widget.TextView;
import android.widget.Toast;

public class JSR268Client extends Activity {

	private static final String TAG = JSR268Client.class.getName();
	private static final String MANAGEMENT_APP = "CardReaderManager.apk";
	private static final String MANAGEMENT_PACKAGE =
			"com.hidglobal.cardreadermanager";
	private static final int REQUEST_APP_INSTALL = 0xbeef;
	private static final String GET_STRING = "Get Card Status in Loop";
	private static final String CANCEL_STRING = "Cancel Get Card Status";
	private static final int STATE_GET = 1;
	private static final int STATE_CANCEL = 2;

	private ICardService mService = null;
	private TerminalFactory mFactory = null;
	private CardTerminal mReader = null;
	
	private TextView mATR = null;
	private Button mCardStatusButton = null;
	private TextView mCardStatus = null;

	private AsyncTask<Void, String, Void> mGetCardStatusTask = null;
	private int mCardStatusButtonState = STATE_GET;

	@Override
	public void onCreate(Bundle savedInstanceState) {
		super.onCreate(savedInstanceState);
		setContentView(R.layout.activity_jsr268_client);

		mATR = (TextView) findViewById(R.id.textView_atr);
		mCardStatusButton = (Button) findViewById(R.id.button1);
		mCardStatus = (TextView) findViewById(R.id.textView_cardStatus);

		/* Set text view default values, set button to default text. */
		resetTextViews();
		mCardStatusButton.setText(GET_STRING);

		/* If the management App is already installed, the service connection
		 * can be established here. Otherwise we have to delay this until the
		 * installation is finished. */
		if (alreadyInstalled(MANAGEMENT_PACKAGE)) {
			mService = CardService.getInstance(this);
		} else {
			/* If the management App cannot be installed, further processing
			 * is impossible. */
			if (!installManagementApp()) {
				showToast("Error: unable to install the management App");
				this.finish();
			}
		}
	}

	@Override
	public void onStop() {
		super.onStop();

		/* Cancel task if not already done by button click. */
		if (mGetCardStatusTask != null) {
			mGetCardStatusTask.cancel(true);
		}
	}

	@Override
	public void onDestroy() {
		super.onDestroy();

		if (mService != null) {
			mService.releaseService();
		}
	}

	@Override
	public void onActivityResult(int requestCode, int resultCode, Intent data) {
		if (requestCode == REQUEST_APP_INSTALL) {
			mService = CardService.getInstance(this);
		}
	}

	@Override
	public boolean onCreateOptionsMenu(Menu menu) {
		getMenuInflater().inflate(R.menu.activity_jsr268_client, menu);
		return true;
	}

	/* Button onClick implementation. */

	public void getCardStatus(View view) {
		/* The button either shows "get" or "cancel", so do the
		 * corresponding work. */

		switch (mCardStatusButtonState) {
		case STATE_GET:
			mGetCardStatusTask = new GetCardStatusTask();
			mGetCardStatusTask.execute();
			mCardStatusButton.setText(CANCEL_STRING);
			mCardStatusButtonState = STATE_CANCEL;
			break;
		case STATE_CANCEL:
			mGetCardStatusTask.cancel(true);
			mCardStatusButton.setText(GET_STRING);
			mCardStatusButtonState = STATE_GET;
			break;
		}
	}

	private void resetTextViews() {
		mATR.setText("");
		mCardStatus.setText("");
	}

	private void setATRString(String string) {
		mATR.setText(string);
	}
	
	private void setStatusString(String string) {
		mCardStatus.setText(string);
	}
	
	private CardTerminal getFirstReader() {
		if (mFactory == null) {
			try {
				mFactory = mService.getTerminalFactory();
			} catch (Exception e) {
				Log.e(TAG, "unable to get terminal factory");
				showToast("Error: unable to get terminal factory");
				return null;
			}
		}

		CardTerminal firstReader = null;
		try {
			/* Get the available card readers as list. */
			List<CardTerminal> readerList = mFactory.terminals().list();
			if (readerList.size() == 0) {
				return null;
			}

			/* Establish a connection with the first reader from the list. */
			firstReader = readerList.get(0);
		} catch (CardException e) {
			Log.e(TAG, e.toString());
			showToast("Error: " + e.toString());
		}
		return firstReader;
	}

	private String byteArrayToString(byte[] array) {
		String hex = "";
		for (int i = 0; i < array.length; i++) {
			hex += "0x" + Integer.toHexString(array[i] & 0x000000ff) + " ";
		}
		return hex;
	}

	private void showToast(String message) {
		Toast toast = Toast.makeText(getApplicationContext(), message,
				Toast.LENGTH_SHORT);
		toast.setGravity(Gravity.CENTER, 0, 0);
		toast.show();
	}

	private boolean alreadyInstalled(String packageName) {
		/* To check whether a package is already installed, to PackageManager
		 * is used, which can be queried about package information. An
		 * exception is thrown if the desired package name is not found.
		 * Package names are fully-qualified, e.g.
		 * "com.example.anExamplePackage". */

		try {
			PackageManager pm = getPackageManager();
			pm.getPackageInfo(packageName, PackageManager.GET_ACTIVITIES);
			return true;
		} catch (PackageManager.NameNotFoundException e) {
			return false;
		}
	}

	private boolean installManagementApp() {
		String cachePath = null;
		try {
			/* Copy the .apk file from the assets directory to the external
			 * cache, from where it can be installed. */
			File temp = File.createTempFile("CardReaderManager", "apk",
					getExternalCacheDir());
			temp.setWritable(true);
			FileOutputStream out = new FileOutputStream(temp);
			InputStream in = getResources().getAssets().open(MANAGEMENT_APP);
			byte[] buffer = new byte[1024];
			int bytes = 0;

			while ((bytes = in.read(buffer)) != -1) {
				out.write(buffer, 0, bytes);
			}
			in.close();
			out.close();
			cachePath = temp.getPath();
		} catch (IOException e) {
			Log.e(TAG, e.toString());
			return false;
		}

		/* Actual installation, calls external Activity that is shown to the
		 * user and returns with call to onActivityResult() to this Activity. */
		Intent promptInstall = new Intent(Intent.ACTION_VIEW);
		promptInstall.setDataAndType(Uri.fromFile(new File(cachePath)),
				"application/vnd.android.package-archive");
		startActivityForResult(promptInstall, REQUEST_APP_INSTALL);
		return true;
	}

	/* This work is done in an AsyncTask (= simple Interface for using Threads
	 * under Android), as the UI must not be blocked, but the status querying
	 * is done in a loop. */
	private class GetCardStatusTask extends AsyncTask<Void, String, Void> {

		@Override
		public Void doInBackground(Void... params) {
			/* Wait until we have the reader instance. */
			while (mReader == null) {
				try {
					Thread.sleep(1000);
				} catch (InterruptedException e) {
				}
				mReader = getFirstReader();
			}

			/* This is done until the button is clicked, which cancels this
			 * AsyncTask. */
			for (; !isCancelled();) {
				try {
					if (mReader.isCardPresent()) {
						/* Connect to the reader. This returns a card object.
						 * "*" indicates that either protocol T=0 or T=1 can be
						 * used. */
						Card card = mReader.connect("*");
						ATR atr = card.getATR();
						card.disconnect(true);
						publishProgress("Card present",
								"ATR: " + byteArrayToString(atr.getBytes()));
					} else {
						publishProgress("No card present",
								"Waiting for card...");
					}
					try {
						/* Don't overtax the USB/Bluetooth connection.*/
						Thread.sleep(1000);
					} catch (InterruptedException e) {
					}
				} catch (CardException e) {
					Log.e(TAG, e.toString());
					publishProgress("Error: " + e.toString(), "");
				}
			}
			return null;
		}

		@Override
		public void onProgressUpdate(String... params) {
			setStatusString(params[0]);
			setATRString(params[1]);
		}

		@Override
		public void onCancelled(Void unused) {
			setStatusString("");
			setATRString("");
		}
	}
}
